# adjustment.py
#
# Copyright 2024 Hari Rana (TheEvilSkeleton)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

from typing import Any, cast

from gi.repository import GObject, Gtk, Gio

from refine.compositions import RefineCompositions
from refine.widgets.reset_button import RefineResetButton

import logging


class RefineAdjustment(Gtk.Adjustment):
    """
    Derivative of `Gtk.Adjustment` implementing the properties needed to work
    with `Gio.Settings`. This widget is meant to be used in conjunction with
    widgets relying on `Gtk.Adjustment`.
    """

    __gtype_name__ = "RefineAdjustment"

    schema: Gio.SettingsSchema

    is_default = GObject.Property(type=bool, default=True)
    is_valid_setting = GObject.Property(type=bool, default=True)
    settings = GObject.Property(type=Gio.Settings)

    def __init__(self, **kwargs: Any) -> None:
        super().__init__(**kwargs)

    def __reset_button_clicked(self, *args: Any) -> None:
        logging.debug("Revert button pressed")
        self.settings.reset(self.key)
        self.set_value(self.settings.get_int(self.key))
        self.__set_default(cast(str, self.key))

    def __set_default(self, key: str) -> None:
        default_int = self.settings.get_default_value(key).get_int32()
        value = int(self.get_value())

        self.is_default = value != default_int

    def __value_changed(self, _adjustment: Gtk.Adjustment, *args: Any) -> None:
        self.__set_default(cast(str, self.key))
        self.settings.set_int(self.key, int(self.get_value()))
        self.value_set(int(self.get_value()))

    @RefineCompositions.schema_id
    def schema_id(self, schema_id: str) -> None: ...

    @RefineCompositions.key
    def key(self, key: str) -> None:
        type Range = tuple[Any, tuple[int, int]]

        schema_key = self.schema.get_key(key)
        variant = cast(Range, schema_key.get_range())
        range = variant[1]

        self.set_lower(range[0])
        self.set_upper(range[1])
        self.set_value(self.settings.get_int(key))

        self.__set_default(key)
        self.connect("value-changed", self.__value_changed)

    def value_set(self, value: Any) -> None:
        """Log key-value pair."""

        logging.debug(f"Set key “{self.key}” to “{value}”")

    @GObject.Property(type=Gtk.Revealer)
    def revealer(self) -> Gtk.Revealer:
        """
        Get or set the `Gtk.Revealer`. If the `Gtk.Revealer` is set, the revealer
        is automatically set up for @self.
        """

        return self._revealer

    @revealer.setter  # type: ignore [no-redef]
    def revealer(self, revealer: Gtk.Revealer) -> None:
        RefineCompositions.set_up_separator_revealer(
            revealer, self.reset_button, "visible"
        )
        self._revealer = revealer

    @GObject.Property(type=RefineResetButton)
    def reset_button(self) -> RefineResetButton:
        """
        Get or set the `RefineResetButton`. If the `RefineResetButton` is set,
        the button is automatically set up for @self.
        """

        return self._reset_button

    @reset_button.setter  # type: ignore [no-redef]
    def reset_button(self, reset_button: RefineResetButton) -> None:
        self.bind_property(
            "is-default", reset_button, "sensitive", GObject.BindingFlags.SYNC_CREATE
        )
        reset_button.connect("clicked", self.__reset_button_clicked)

        self._reset_button = reset_button
